package com.arcsoft.face;

import com.arcsoft.face.enums.*;
import com.arcsoft.face.toolkit.ImageInfo;
import com.arcsoft.face.toolkit.ImageInfoEx;
import java.io.File;
import java.util.LinkedList;
import java.util.List;

/**
 * @author st7251
 */
public class FaceEngine {


    private native int nativeInitFaceEngine(long detectMode, int detectFaceOrientPriority, int detectFaceMaxNum, int combinedMask);

    private native int nativeActiveOnlineFaceEngine(String appId, String sdkKey, String activeKey);

    private native int nativeActiveOfflineFaceEngine(String filePath);

    private native int nativeGetActiveFile(ActiveFileInfo activeFileInfo);

    private native int nativeUnInitFaceEngine(long handle);

    private native int nativeGetVersion(VersionInfo versionInfo);

    private native int nativeSetFaceAttributeParam(long handle, AttributeParam attributeParam);

    private native int nativeDetectFaces(long handle, byte[] imageData, int width, int height, int format, int detectModel, FaceInfo[] faceInfoArray);

    private native int nativeDetectFacesEx(long handle, byte[][] imageDataPlanes, int[] imageStrides, int width, int height, int format, int detectModel, FaceInfo[] faceInfoArray);

    private native int nativeUpdateFaceData(long handle, byte[] imageData, int width, int height, int format, FaceInfo[] faceInfoArray);

    private native int nativeUpdateFaceDataEx(long handle, byte[][] imageDataPlanes, int[] imageStrides, int width, int height, int format, FaceInfo[] faceInfoArray);

    private native int nativeSetLivenessParam(long handle, float rgbThreshold, float irThreshold,float fqThreshold);

    private native int nativeGetLivenessParam(long handle,LivenessParam livenessParam);

    private native int nativeProcess(long handle, byte[] imageData, int width, int height, int format, FaceInfo[] faceInfoArray, int combineMask);

    private native int nativeProcessEx(long handle, byte[][] imageDataPlanes, int[] imageStrides, int width, int height, int format, FaceInfo[] faceInfoArray, int combineMask);

    private native int nativeProcessIr(long handle, byte[] imageData, int width, int height, int format, FaceInfo[] faceInfoArray, int combineMask);

    private native int nativeProcessIrEx(long handle, byte[][] imageDataPlanes, int[] imageStrides, int width, int height, int format, FaceInfo[] faceInfoArray, int combineMask);

    private native int nativeExtractFaceFeature(long handle, byte[] imageData, int width, int height, int format, Rect faceRect, int degree, byte[] faceData, int faceDataSize, int extractType, int mask, byte[] featureData);

    private native int nativeExtractFaceFeatureEx(long handle, byte[][] imageDataPlanes, int[] imageStrides, int width, int height, int format, Rect faceRect, int degree, byte[] faceData, int faceDataSize, int extractType, int mask, byte[] featureData);

    private native int nativeFaceFeatureCompare(long handle, byte[] feature1, byte[] feature2, int compareModel, FaceSimilar faceSimilar);

    private native int nativeFaceSearch(long handle, byte[] faceFeature, int compareModel, SearchResult searchResult);

    private native int nativeRegisterFaceFeatureArray(long handle, FaceFeatureInfo[] faceFeatureInfos);

    private native int nativeRemoveFaceFeature(long handle, int searchId);

    private native int nativeUpdateFaceFeature(long handle, FaceFeatureInfo faceFeatureInfo);

    private native int nativeGetFaceFeature(long handle, int searchId, FaceFeatureInfo faceFeatureInfo);

    private native int nativeGetFaceCount(long handle, FaceSearchCount faceSearchCount);

    private native int nativeGetAge(long handle, AgeInfo[] ageInfoArray);

    private native int nativeGetGender(long handle, GenderInfo[] genderInfoArray);

    private native int nativeGetLiveness(long handle, LivenessInfo[] livenessInfos);

    private native int nativeGetLivenessIr(long handle, IrLivenessInfo[] livenessInfos);

    private native int nativeGetMask(long handle, MaskInfo[] maskInfos);

    private native int nativeGetActiveDeviceInfo(ActiveDeviceInfo activeDeviceInfo);

    private native int nativeImageQualityDetect(long handle, byte[] imageData, int width, int height, int format, Rect faceRect, int orient, byte[] faceData, int faceDataSize, int mask, ImageQuality imageQuality);

    private native int nativeImageQualityDetectEx(long handle, byte[][] imageDataPlanes, int[] imageStrides, int width, int height, int format, Rect faceRect, int orient, byte[] faceData, int faceDataSize, int mask, ImageQuality imageQuality);


    /**
     * 初始化和调用引擎的属性，人脸检测
     */
    private final int ASF_FACE_DETECT = 0x00000001;
    /**
     * 初始化和调用引擎的属性，人脸识别
     */
    private final int ASF_FACE_RECOGNITION = 0x00000004;
    /**
     * 初始化和调用引擎的属性，年龄检测
     */
    private final int ASF_AGE = 0x00000008;
    /**
     * 初始化和调用引擎的属性，性别检测
     */
    private final int ASF_GENDER = 0x00000010;

    /**
     * 初始化和调用引擎的属性，RGB活体检测
     */
    private final int ASF_LIVENESS = 0x00000080;

    /**
     * 初始化和调用引擎的属性，IR活体检测
     */
    private final int ASF_IR_LIVENESS = 0x00000400;

    /**
     * 初始化和调用引擎的属性，图像质量检测
     */
    private final int ASF_IMAGEQUALITY = 0x00000200;

    /**
     * 初始化和调用引擎的属性，口罩检测
     */
    private final int ASF_MASKDETECT = 0x00001000;

    /**
     * 初始化和调用引擎的属性，人脸信息更新
     */
    private final int ASF_UPDATE_FACEDATA = 0x00002000;

    /**
     * 引擎句柄
     */
    private long handle = 0L;

    /**
     * 最大检查的人脸个数
     */
    private int detectFaceMaxNum;

    private static final List<String> libCache = new LinkedList<String>();

    private static final String WIN_FACE_ENGINE_JNI_LIBRARY = "libarcsoft_face_engine_jni";
    private static final String LINUX_FACE_ENGINE_JNI_LIBRARY = "arcsoft_face_engine_jni";


    private static synchronized void libraryLoad(String libPath) {
        if (libPath == null || libPath.length() == 0) {
            if (!libCache.contains(WIN_FACE_ENGINE_JNI_LIBRARY)) {
                String os = System.getProperty("os.name");
                if (os.toLowerCase().startsWith("win")) {
                    System.loadLibrary(WIN_FACE_ENGINE_JNI_LIBRARY);
                } else {
                    System.loadLibrary(LINUX_FACE_ENGINE_JNI_LIBRARY);
                }
                libCache.add(WIN_FACE_ENGINE_JNI_LIBRARY);
            }
        } else {
            if (libPath.endsWith("/") || libPath.endsWith("\\")) {
                libPath = libPath.substring(0, libPath.length() - 1);
            }
            libPath += File.separator;
            if (!libCache.contains(libPath)) {
                String os = System.getProperty("os.name");
                if (os.toLowerCase().startsWith("win")) {
                    System.load(libPath + "libarcsoft_face.dll");
                    System.load(libPath + "libarcsoft_face_engine.dll");
                    System.load(libPath + "libarcsoft_face_engine_jni.dll");
                } else {
                    System.load(libPath + "libarcsoft_face.so");
                    System.load(libPath + "libarcsoft_face_engine.so");
                    System.load(libPath + "libarcsoft_face_engine_jni.so");
                }
                libCache.add(libPath);
            }

        }
    }

    /**
     * 创建引擎
     */
    public FaceEngine() {
        libraryLoad(null);
    }

    /**
     * 创建引擎
     *
     * @param libPath 引擎库路径
     */
    public FaceEngine(String libPath) {
        libraryLoad(libPath);
    }

    /**
     * 激活SDK,与activeOnline功能一致，建议用activeOnline
     *
     * @param appId     官网申请的APP_ID
     * @param sdkKey    官网申请的SDK_KEY
     * @param activeKey 官网申请的ACTIVE_KEY
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int activeOnline(String appId, String sdkKey, String activeKey) {
        if (appId == null) {
            appId = "";
        }
        if (sdkKey == null) {
            sdkKey = "";
        }
        if (activeKey == null) {
            activeKey = "";
        }
        int resultCode = nativeActiveOnlineFaceEngine(appId, sdkKey, activeKey);
        return resultCode;
    }


    /**
     * 离线激活SDK
     *
     * @param filePath 激活文件路径
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int activeOffline(String filePath) {
        if (filePath == null) {
            filePath = "";
        }
        int resultCode = nativeActiveOfflineFaceEngine(filePath);
        return resultCode;
    }

    /**
     * 获取激活文件信息
     *
     * @param activeFileInfo 激活文件信息对象，用于输出激活文件信息数据
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int getActiveFileInfo(ActiveFileInfo activeFileInfo) {
        int errorCode = nativeGetActiveFile(activeFileInfo);
        return errorCode;
    }

    /**
     * 初始化人脸引擎
     *
     * @param engineConfiguration 引擎配置类
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int init(EngineConfiguration engineConfiguration) {
        if (engineConfiguration == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        long detectMode = engineConfiguration.getDetectMode().getValue();
        int detectFaceOrientPriority = engineConfiguration.getDetectFaceOrientPriority().getValue();
        int detectFaceMaxNum = engineConfiguration.getDetectFaceMaxNum();
        this.detectFaceMaxNum = detectFaceMaxNum;
        int combinedMask = 0;
        if (engineConfiguration.getFunctionConfiguration().isSupportFaceDetect()) {
            combinedMask |= ASF_FACE_DETECT;
        }
        if (engineConfiguration.getFunctionConfiguration().isSupportFaceRecognition()) {
            combinedMask |= ASF_FACE_RECOGNITION;
        }
        if (engineConfiguration.getFunctionConfiguration().isSupportAge()) {
            combinedMask |= ASF_AGE;
        }
        if (engineConfiguration.getFunctionConfiguration().isSupportGender()) {
            combinedMask |= ASF_GENDER;
        }
        if (engineConfiguration.getFunctionConfiguration().isSupportLiveness()) {
            combinedMask |= ASF_LIVENESS;
        }
        if (engineConfiguration.getFunctionConfiguration().isSupportIRLiveness()) {
            combinedMask |= ASF_IR_LIVENESS;
        }
        if (engineConfiguration.getFunctionConfiguration().isSupportImageQuality()) {
            combinedMask |= ASF_IMAGEQUALITY;
        }
        if (engineConfiguration.getFunctionConfiguration().isSupportMaskDetect()) {
            combinedMask |= ASF_MASKDETECT;
        }
        if (engineConfiguration.getFunctionConfiguration().isSupportUpdateFaceData()) {
            combinedMask |= ASF_UPDATE_FACEDATA;
        }

        int errorCode = nativeInitFaceEngine(detectMode, detectFaceOrientPriority, detectFaceMaxNum, combinedMask);
        return errorCode;
    }

    /**
     * 销毁引擎
     *
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int unInit() {
        if (handle != 0L) {
            int errorCode = nativeUnInitFaceEngine(handle);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * 创建活体阈值参数对象
     *
     * @param rgbThreshold RGB活体阈值
     * @param irThreshold  IR活体阈值
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int setLivenessParam(float rgbThreshold, float irThreshold,float fqThreshold) {
        if (handle != 0L) {
            int errorCode = nativeSetLivenessParam(handle, rgbThreshold, irThreshold,fqThreshold);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * 获取活体设置参数
     * @param livenessParam
     * @return
     */
    public int getLivenessParam(LivenessParam livenessParam) {
        if (livenessParam==null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        if (handle != 0L) {
            int errorCode = nativeGetLivenessParam(handle, livenessParam);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * 设置人脸 属性阈值
     *
     * @param attributeParam 属性阈值
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int setFaceAttributeParam(AttributeParam attributeParam) {
        if (handle != 0L) {
            int errorCode = nativeSetFaceAttributeParam(handle, attributeParam);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }


    /**
     * 人脸检测
     *
     * @param data         输入的图像数据
     * @param width        图片宽度，为4的倍数
     * @param height       图片高度，YUYV/I420/NV21/NV12格式为2的倍数；BGR24/GRAY/DEPTH_U16格式无限制
     * @param imageFormat  图像的颜色空间格式
     * @param faceInfoList 人脸列表，传入后赋值
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int detectFaces(byte[] data, int width, int height, ImageFormat imageFormat, List<FaceInfo> faceInfoList) {
        return detectFaces(data, width, height, imageFormat, DetectModel.ASF_DETECT_MODEL_RGB, faceInfoList);
    }

    /**
     * 人脸检测
     *
     * @param imageInfo    图像信息
     * @param faceInfoList 人脸输出列表
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int detectFaces(ImageInfo imageInfo, List<FaceInfo> faceInfoList) {
        if (imageInfo == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        return detectFaces(imageInfo.getImageData(), imageInfo.getWidth(), imageInfo.getHeight(), imageInfo.getImageFormat(), DetectModel.ASF_DETECT_MODEL_RGB, faceInfoList);
    }

    /**
     * 人脸检测
     *
     * @param data         输入的图像数据
     * @param width        图片宽度，为4的倍数
     * @param height       图片高度，YUYV/I420/NV21/NV12格式为2的倍数；BGR24/GRAY/DEPTH_U16格式无限制
     * @param imageFormat  图像的颜色空间格式
     * @param detectModel  人脸检查模型
     * @param faceInfoList 人脸列表，传入后赋值
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int detectFaces(byte[] data, int width, int height, ImageFormat imageFormat, DetectModel detectModel, List<FaceInfo> faceInfoList) {
        if (data == null || imageFormat == null || detectModel == null || faceInfoList == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(data, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }


        faceInfoList.clear();
        if (handle != 0L) {
            FaceInfo[] faceInfoArray = new FaceInfo[detectFaceMaxNum];
            int code = nativeDetectFaces(handle, data, width, height, imageFormat.getValue(), detectModel.getValue(), faceInfoArray);
            for (FaceInfo faceInfo : faceInfoArray) {
                if (faceInfo == null) break;
                faceInfoList.add(faceInfo);
            }
            return code;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * 图像信息
     *
     * @param imageInfoEx  图像信息
     * @param faceInfoList 人脸输出列表
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int detectFaces(ImageInfoEx imageInfoEx, List<FaceInfo> faceInfoList) {
        if (imageInfoEx == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        return detectFaces(imageInfoEx.getImageDataPlanes(),
                imageInfoEx.getImageStrides(), imageInfoEx.getWidth(),
                imageInfoEx.getHeight(), imageInfoEx.getImageFormat(), DetectModel.ASF_DETECT_MODEL_RGB, faceInfoList);

    }

    /**
     * 图像信息
     *
     * @param imageInfoEx  图像信息
     * @param detectModel  人脸检测模型
     * @param faceInfoList 人脸输出列表
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int detectFaces(ImageInfoEx imageInfoEx, DetectModel detectModel, List<FaceInfo> faceInfoList) {
        if (imageInfoEx == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        return detectFaces(imageInfoEx.getImageDataPlanes(),
                imageInfoEx.getImageStrides(), imageInfoEx.getWidth(),
                imageInfoEx.getHeight(), imageInfoEx.getImageFormat(), detectModel, faceInfoList);

    }

    /**
     * 人脸检测
     *
     * @param imageDataPlanes 人脸图像通道
     * @param imageStrides    步长
     * @param width           图片宽度，为4的倍数
     * @param height          图片高度，YUYV/I420/NV21/NV12格式为2的倍数；BGR24/GRAY/DEPTH_U16格式无限制
     * @param imageFormat     图像的颜色空间格式
     * @param detectModel     人脸检查模型
     * @param faceInfoList    人脸列表，传入后赋值
     * @return 错误码，详细见({@link ErrorInfo})
     */
    private int detectFaces(byte[][] imageDataPlanes, int[] imageStrides, int width, int height, ImageFormat imageFormat, DetectModel detectModel, List<FaceInfo> faceInfoList) {
        if (imageDataPlanes == null || imageFormat == null || detectModel == null || faceInfoList == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(imageDataPlanes, imageStrides, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }


        faceInfoList.clear();

        if (handle != 0L) {
            FaceInfo[] faceInfoArray = new FaceInfo[detectFaceMaxNum];

            int errorCode = nativeDetectFacesEx(handle, imageDataPlanes, imageStrides, width, height, imageFormat.getValue(), detectModel.getValue(), faceInfoArray);

            for (FaceInfo faceInfo : faceInfoArray) {
                if (faceInfo == null) break;
                faceInfoList.add(faceInfo);
            }
            return errorCode;

        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }


    /**
     * 更新人脸信息
     *
     * @param faceInfoList 人脸输出列表
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int updateFaceData(ImageInfo imageInfo, List<FaceInfo> faceInfoList) {
        if (imageInfo == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        return updateFaceData(imageInfo.getImageData(), imageInfo.getWidth(), imageInfo.getHeight(), imageInfo.getImageFormat(), faceInfoList);
    }

    /**
     * 更新人脸信息
     *
     * @param imageInfoEx  图像信息
     * @param faceInfoList 人脸输出列表
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int updateFaceData(ImageInfoEx imageInfoEx, List<FaceInfo> faceInfoList) {
        if (imageInfoEx == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        return updateFaceData(imageInfoEx.getImageDataPlanes(),
                imageInfoEx.getImageStrides(), imageInfoEx.getWidth(),
                imageInfoEx.getHeight(), imageInfoEx.getImageFormat(), faceInfoList);

    }


    /**
     * 更新人脸信息
     *
     * @param data         输入的图像数据
     * @param width        图片宽度，为4的倍数
     * @param height       图片高度，YUYV/I420/NV21/NV12格式为2的倍数；BGR24/GRAY/DEPTH_U16格式无限制
     * @param imageFormat  图像的颜色空间格式
     * @param faceInfoList 人脸列表，传入后赋值
     * @return 错误码，详细见({@link ErrorInfo})
     */
    private int updateFaceData(byte[] data, int width, int height, ImageFormat imageFormat, List<FaceInfo> faceInfoList) {
        if (data == null || imageFormat == null || faceInfoList == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(data, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }


        if (handle != 0L) {
            FaceInfo[] faceInfoArray = faceInfoList.toArray(new FaceInfo[0]);

            int code = nativeUpdateFaceData(handle, data, width, height, imageFormat.getValue(), faceInfoArray);
            return code;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * @param imageDataPlanes 人脸图像通道
     * @param imageStrides    步长
     * @param width           图片宽度，为4的倍数
     * @param height          图片高度，YUYV/I420/NV21/NV12格式为2的倍数；BGR24/GRAY/DEPTH_U16格式无限制
     * @param imageFormat     图像的颜色空间格式
     * @param faceInfoList    人脸列表，传入后赋值
     * @return 错误码，详细见({@link ErrorInfo})
     */
    private int updateFaceData(byte[][] imageDataPlanes, int[] imageStrides, int width, int height, ImageFormat imageFormat, List<FaceInfo> faceInfoList) {
        if (imageDataPlanes == null || imageFormat == null || faceInfoList == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(imageDataPlanes, imageStrides, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }

        if (handle != 0L) {
            FaceInfo[] faceInfoArray = faceInfoList.toArray(new FaceInfo[0]);

            int errorCode = nativeUpdateFaceDataEx(handle, imageDataPlanes, imageStrides, width, height, imageFormat.getValue(), faceInfoArray);
            return errorCode;

        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * RGB活体、年龄、性别、三维角度检测，在调用该函数后，可以调用getLiveness(List) ，getAge(List)，getGender(List)，getFace3DAngle(List)等分别获取 RGB活体、年龄、性别、三维角度等检测结果； RGB活体最多支持 1 个人脸信息的检测，超过部分返回未知； 年龄、性别、三维角度最多支持4个人脸信息的检测，超过部分返回未知
     *
     * @param imageInfo             图像数据
     * @param faceInfoList          人脸列表
     * @param functionConfiguration 功能配置组合
     * @return
     */
    public int process(ImageInfo imageInfo, List<FaceInfo> faceInfoList, FunctionConfiguration functionConfiguration) {
        if (imageInfo == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        return process(imageInfo.getImageData(), imageInfo.getWidth(), imageInfo.getHeight(), imageInfo.getImageFormat()
                , faceInfoList, functionConfiguration);
    }

    /**
     * RGB活体、年龄、性别、三维角度检测，在调用该函数后，可以调用getLiveness(List) ，getAge(List)，getGender(List)，getFace3DAngle(List)等分别获取 RGB活体、年龄、性别、三维角度等检测结果； RGB活体最多支持 1 个人脸信息的检测，超过部分返回未知； 年龄、性别、三维角度最多支持4个人脸信息的检测，超过部分返回未知
     *
     * @param data                  输入的图像数据
     * @param width                 图片宽度，为4的倍数
     * @param height                图片高度，YUYV/I420/NV21/NV12格式为2的倍数，BGR24格式无限制
     * @param imageFormat           图像的颜色空间格式
     * @param faceInfoList          人脸列表
     * @param functionConfiguration 功能配置组合
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int process(byte[] data, int width, int height, ImageFormat imageFormat, List<FaceInfo> faceInfoList, FunctionConfiguration functionConfiguration) {
        if (faceInfoList == null
                || data == null
                || imageFormat == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(data, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }


        int _combinedMask = 0;
        if (functionConfiguration.isSupportFaceDetect()) {
            _combinedMask |= ASF_FACE_DETECT;
        }
        if (functionConfiguration.isSupportFaceRecognition()) {
            _combinedMask |= ASF_FACE_RECOGNITION;
        }
        if (functionConfiguration.isSupportAge()) {
            _combinedMask |= ASF_AGE;
        }
        if (functionConfiguration.isSupportGender()) {
            _combinedMask |= ASF_GENDER;
        }
        if (functionConfiguration.isSupportLiveness()) {
            _combinedMask |= ASF_LIVENESS;
        }
        if (functionConfiguration.isSupportIRLiveness()) {
            _combinedMask |= ASF_IR_LIVENESS;
        }
        if (functionConfiguration.isSupportImageQuality()) {
            _combinedMask |= ASF_IMAGEQUALITY;
        }
        if (functionConfiguration.isSupportMaskDetect()) {
            _combinedMask |= ASF_MASKDETECT;
        }
        if (functionConfiguration.isSupportUpdateFaceData()) {
            _combinedMask |= ASF_UPDATE_FACEDATA;
        }

        if (handle != 0L) {
            int errorCode = nativeProcess(handle, data, width, height, imageFormat.getValue(), faceInfoList.toArray(new FaceInfo[0]), _combinedMask);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * @param imageInfoEx           图像信息
     * @param faceInfoList          人脸列表
     * @param functionConfiguration 功能配置组合
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int process(ImageInfoEx imageInfoEx, List<FaceInfo> faceInfoList, FunctionConfiguration functionConfiguration) {
        if (imageInfoEx == null
                || faceInfoList == null
                || functionConfiguration == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        return process(imageInfoEx.getImageDataPlanes(), imageInfoEx.getImageStrides(),
                imageInfoEx.getWidth(), imageInfoEx.getHeight(), imageInfoEx.getImageFormat(), faceInfoList, functionConfiguration);
    }

    /**
     * @param imageDataPlanes 人脸图像通道
     * @param imageStrides    步长
     * @param width           图片宽度，为4的倍数
     * @param height          图片高度，YUYV/I420/NV21/NV12格式为2的倍数；BGR24/GRAY/DEPTH_U16格式无限制
     * @param imageFormat     图像的颜色空间格式
     * @param faceInfoList    人脸列表，传入后赋值
     * @return 错误码，详细见({@link ErrorInfo})
     */
    private int process(byte[][] imageDataPlanes, int[] imageStrides, int width, int height, ImageFormat imageFormat, List<FaceInfo> faceInfoList, FunctionConfiguration functionConfiguration) {
        if (faceInfoList == null
                || imageDataPlanes == null
                || imageStrides == null
                || imageFormat == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(imageDataPlanes, imageStrides, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }


        int _combinedMask = 0;
        if (functionConfiguration.isSupportFaceDetect()) {
            _combinedMask |= ASF_FACE_DETECT;
        }
        if (functionConfiguration.isSupportFaceRecognition()) {
            _combinedMask |= ASF_FACE_RECOGNITION;
        }
        if (functionConfiguration.isSupportAge()) {
            _combinedMask |= ASF_AGE;
        }
        if (functionConfiguration.isSupportGender()) {
            _combinedMask |= ASF_GENDER;
        }
        if (functionConfiguration.isSupportLiveness()) {
            _combinedMask |= ASF_LIVENESS;
        }
        if (functionConfiguration.isSupportIRLiveness()) {
            _combinedMask |= ASF_IR_LIVENESS;
        }
        if (functionConfiguration.isSupportImageQuality()) {
            _combinedMask |= ASF_IMAGEQUALITY;
        }
        if (functionConfiguration.isSupportMaskDetect()) {
            _combinedMask |= ASF_MASKDETECT;
        }
        if (functionConfiguration.isSupportUpdateFaceData()) {
            _combinedMask |= ASF_UPDATE_FACEDATA;
        }

        if (handle != 0L) {
            int errorCode = nativeProcessEx(handle, imageDataPlanes, imageStrides, width, height, imageFormat.getValue(), faceInfoList.toArray(new FaceInfo[0]), _combinedMask);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * RGB活体、年龄、性别、三维角度检测，在调用该函数后，可以调用getLiveness(List) ，getAge(List)，getGender(List)，getFace3DAngle(List)等分别获取 RGB活体、年龄、性别、三维角度等的检测结果； RGB活体最多支持 1 个人脸信息的检测，超过部分返回未知； 年龄、性别、三维角度最多支持4个人脸信息的检测，超过部分返回未知
     *
     * @param imageInfo    图像数据
     * @param faceInfoList 人脸列表
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int processIr(ImageInfo imageInfo, List<FaceInfo> faceInfoList, FunctionConfiguration functionConfiguratio) {
        if (faceInfoList == null
                || imageInfo == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        int errorCode = processIr(imageInfo.getImageData(), imageInfo.getWidth(), imageInfo.getHeight(), imageInfo.getImageFormat(), faceInfoList, functionConfiguratio);
        return errorCode;

    }

    /**
     * RGB活体、年龄、性别、三维角度检测，在调用该函数后，可以调用getLiveness(List) ，getAge(List)，getGender(List)，getFace3DAngle(List)分别获取 RGB活体、年龄、性别、三维角度的检测结果； RGB活体最多支持 1 个人脸信息的检测，超过部分返回未知； 年龄、性别、三维角度最多支持4个人脸信息的检测，超过部分返回未知
     *
     * @param data                  输入的图像数据
     * @param width                 图片宽度，为4的倍数
     * @param height                图片高度，YUYV/I420/NV21/NV12格式为2的倍数，BGR24格式无限制
     * @param imageFormat           图像的颜色空间格式
     * @param faceInfoList          人脸列表
     * @param functionConfiguration 功能配置组合
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int processIr(byte[] data, int width, int height, ImageFormat imageFormat, List<FaceInfo> faceInfoList, FunctionConfiguration functionConfiguration) {
        if (faceInfoList == null
                || data == null
                || imageFormat == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(data, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }

        int _combinedMask = 0;

        if (functionConfiguration.isSupportFaceDetect()) {
            _combinedMask |= ASF_FACE_DETECT;
        }
        if (functionConfiguration.isSupportFaceRecognition()) {
            _combinedMask |= ASF_FACE_RECOGNITION;
        }
        if (functionConfiguration.isSupportAge()) {
            _combinedMask |= ASF_AGE;
        }
        if (functionConfiguration.isSupportGender()) {
            _combinedMask |= ASF_GENDER;
        }
        if (functionConfiguration.isSupportLiveness()) {
            _combinedMask |= ASF_LIVENESS;
        }
        if (functionConfiguration.isSupportIRLiveness()) {
            _combinedMask |= ASF_IR_LIVENESS;
        }
        if (functionConfiguration.isSupportImageQuality()) {
            _combinedMask |= ASF_IMAGEQUALITY;
        }

        if (handle != 0L) {
            int errorCode = nativeProcessIr(handle, data, width, height, imageFormat.getValue(), faceInfoList.toArray(new FaceInfo[0]), _combinedMask);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * @param imageInfoEx           图像信息
     * @param faceInfoList          人脸列表
     * @param functionConfiguration 功能配置组合
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int processIr(ImageInfoEx imageInfoEx, List<FaceInfo> faceInfoList, FunctionConfiguration functionConfiguration) {
        if (imageInfoEx == null
                || faceInfoList == null
                || functionConfiguration == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        return processIr(imageInfoEx.getImageDataPlanes(), imageInfoEx.getImageStrides(),
                imageInfoEx.getWidth(), imageInfoEx.getHeight(), imageInfoEx.getImageFormat(), faceInfoList, functionConfiguration);
    }

    /**
     * @param imageDataPlanes       人脸图像通道
     * @param imageStrides          步长
     * @param width                 图片宽度，为4的倍数
     * @param height                图片高度，YUYV/I420/NV21/NV12格式为2的倍数；BGR24/GRAY/DEPTH_U16格式无限制
     * @param imageFormat           图像的颜色空间格式
     * @param faceInfoList          人脸列表，传入后赋值
     * @param functionConfiguration 功能类型
     * @return 错误码，详细见({@link ErrorInfo})
     */
    private int processIr(byte[][] imageDataPlanes, int[] imageStrides, int width, int height, ImageFormat imageFormat, List<FaceInfo> faceInfoList, FunctionConfiguration functionConfiguration) {
        if (faceInfoList == null
                || imageDataPlanes == null
                || imageStrides == null
                || imageFormat == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(imageDataPlanes, imageStrides, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }


        int _combinedMask = 0;

        if (functionConfiguration.isSupportFaceDetect()) {
            _combinedMask |= ASF_FACE_DETECT;
        }
        if (functionConfiguration.isSupportFaceRecognition()) {
            _combinedMask |= ASF_FACE_RECOGNITION;
        }
        if (functionConfiguration.isSupportAge()) {
            _combinedMask |= ASF_AGE;
        }
        if (functionConfiguration.isSupportGender()) {
            _combinedMask |= ASF_GENDER;
        }
        if (functionConfiguration.isSupportLiveness()) {
            _combinedMask |= ASF_LIVENESS;
        }
        if (functionConfiguration.isSupportIRLiveness()) {
            _combinedMask |= ASF_IR_LIVENESS;
        }
        if (functionConfiguration.isSupportImageQuality()) {
            _combinedMask |= ASF_IMAGEQUALITY;
        }

        if (handle != 0L) {
            int errorCode = nativeProcessIrEx(handle, imageDataPlanes, imageStrides, width, height, imageFormat.getValue(), faceInfoList.toArray(new FaceInfo[0]), _combinedMask);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * @param faceInfo 人脸信息
     * @param feature  人脸特征对象，用于输出人脸特征数据
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int extractFaceFeature(ImageInfo imageInfo, FaceInfo faceInfo, ExtractType extractType, int mask, FaceFeature feature) {
        if (imageInfo == null || faceInfo == null || feature == null || extractType == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        return extractFaceFeature(imageInfo.getImageData(), imageInfo.getWidth(),
                imageInfo.getHeight(), imageInfo.getImageFormat(), faceInfo, extractType, mask, feature);

    }

    /**
     * 提取人脸特征数据
     *
     * @param data        图像数据
     * @param width       图片宽度，为4的倍数
     * @param height      图片高度，YUYV/I420/NV21/NV12格式为2的倍数；BGR24/GRAY格式无限制
     * @param imageFormat 图像的颜色空间格式
     * @param faceInfo    人脸信息
     * @param feature     人脸特征对象，用于输出人脸特征数据
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int extractFaceFeature(byte[] data, int width, int height, ImageFormat imageFormat, FaceInfo faceInfo, ExtractType extractType, int mask, FaceFeature feature) {

        if (feature == null || imageFormat == null
                || feature.getFeatureData() == null
                || feature.getFeatureData().length < FaceFeature.FEATURE_SIZE
                || faceInfo == null
                || data == null || extractType == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(data, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }


        if (handle != 0L) {
            int errorCode = nativeExtractFaceFeature(handle, data, width, height, imageFormat.getValue(), faceInfo.getRect(), faceInfo.getOrient(), faceInfo.getFaceData(), FaceInfo.FACE_DATA_SIZE, extractType.getExtractType(), mask, feature.featureData);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }

    }


    /**
     * @param imageInfoEx 图像信息
     * @param faceInfo    人脸信息
     * @param feature     人脸特征对象，用于输出人脸特征数据
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int extractFaceFeature(ImageInfoEx imageInfoEx, FaceInfo faceInfo, ExtractType extractType, int mask, FaceFeature feature) {
        if (imageInfoEx == null || faceInfo == null || feature == null || extractType == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        return extractFaceFeature(imageInfoEx.getImageDataPlanes(), imageInfoEx.getImageStrides(), imageInfoEx.getWidth(),
                imageInfoEx.getHeight(), imageInfoEx.getImageFormat(), faceInfo, extractType, mask, feature);

    }

    /**
     * @param imageDataPlanes 图像通道
     * @param imageStrides    图像步长
     * @param width           宽度
     * @param height          高度
     * @param imageFormat     格式
     * @param faceInfo        人脸信息
     * @param feature         人脸特征对象，用于输出人脸特征数据
     * @return 错误码，详细见({@link ErrorInfo})
     */
    private int extractFaceFeature(byte[][] imageDataPlanes, int[] imageStrides, int width, int height, ImageFormat imageFormat, FaceInfo faceInfo, ExtractType extractType, int mask, FaceFeature feature) {

        if (feature == null || imageFormat == null
                || feature.getFeatureData() == null
                || faceInfo == null
                || imageDataPlanes == null
                || imageStrides == null
                || extractType == null
        ) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(imageDataPlanes, imageStrides, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }


        if (handle != 0L) {
            int errorCode = nativeExtractFaceFeatureEx(handle, imageDataPlanes, imageStrides, width, height, imageFormat.getValue(), faceInfo.getRect(), faceInfo.getOrient(), faceInfo.getFaceData(), FaceInfo.FACE_DATA_SIZE, extractType.getExtractType(), mask, feature.featureData);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * 比对人脸特征数据获取相似度
     *
     * @param feature1    人脸特征对象
     * @param feature2    人脸特征对象
     * @param faceSimilar 相似度信息，用于输出相似度数据
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int compareFaceFeature(FaceFeature feature1, FaceFeature feature2, FaceSimilar faceSimilar) {
        return compareFaceFeature(feature1, feature2, CompareModel.LIFE_PHOTO, faceSimilar);
    }

    /**
     * 比对人脸特征数据获取相似度
     *
     * @param feature1     人脸特征对象
     * @param feature2     人脸特征对象
     * @param compareModel 比对模型，生活照比对还是身份证比对
     * @param faceSimilar  相似度信息，用于输出相似度数据
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int compareFaceFeature(FaceFeature feature1, FaceFeature feature2, CompareModel compareModel, FaceSimilar faceSimilar) {
        if (feature1 == null || feature1.getFeatureData() == null || feature2 == null || feature2.getFeatureData() == null || faceSimilar == null || compareModel == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (handle != 0L) {
            int errorCode = nativeFaceFeatureCompare(handle, feature1.featureData, feature2.featureData, compareModel.getValue(), faceSimilar);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }

    }

    /**
     * 搜索人脸特征
     * @param faceFeature 待比对的人脸特征
     * @param searchResult 返回的搜索的结果
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int searchFaceFeature(FaceFeature faceFeature, SearchResult searchResult) {
        return this.searchFaceFeature(faceFeature, CompareModel.LIFE_PHOTO, searchResult);
    }


    /**
     * 搜索人脸特征
     * @param faceFeature 待比对的人脸特征
     * @param compareModel 搜索模式
     * @param searchResult 返回的搜索的结果
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int searchFaceFeature(FaceFeature faceFeature, CompareModel compareModel, SearchResult searchResult) {

        if (faceFeature == null || faceFeature.getFeatureData() == null || compareModel == null || searchResult == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (handle == 0L) {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }

        return nativeFaceSearch(handle, faceFeature.featureData, compareModel.getValue(), searchResult);
    }

    /**
     * 注册人脸特征
     * @param faceFeatureInfo 人脸信息
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int registerFaceFeature(FaceFeatureInfo faceFeatureInfo) {
        if (faceFeatureInfo == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        if (faceFeatureInfo.getFeatureData() == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        if (handle == 0L) {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
        return nativeRegisterFaceFeatureArray(handle, new FaceFeatureInfo[]{faceFeatureInfo});
    }

    /**
     * 批量注册人脸特征
     * @param faceFeatureInfoList 人脸信息列表
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int registerFaceFeature(List<FaceFeatureInfo> faceFeatureInfoList) {
        if (faceFeatureInfoList == null || faceFeatureInfoList.isEmpty()) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        for (FaceFeatureInfo faceFeatureInfo : faceFeatureInfoList) {
            if (faceFeatureInfo.getFeatureData() == null || faceFeatureInfo.getFeatureData().length != FaceFeature.FEATURE_SIZE) {
                return ErrorInfo.MERR_INVALID_PARAM.getValue();
            }
        }
        if (handle == 0L) {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }

        return nativeRegisterFaceFeatureArray(handle, faceFeatureInfoList.toArray(new FaceFeatureInfo[0]));
    }

    /**
     * 删除人脸特征
     * @param searchId 人脸Id
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int removeFaceFeature(int searchId) {
        if (handle == 0L) {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
        return nativeRemoveFaceFeature(handle, searchId);
    }

    /**
     *  更新人脸特征
     * @param faceFeatureInfo 人脸信息
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int updateFaceFeature(FaceFeatureInfo faceFeatureInfo) {
        if (faceFeatureInfo == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
//        if (faceFeatureInfo.getFeatureData() == null || faceFeatureInfo.getFeatureData().length != FaceFeature.FEATURE_SIZE) {
//            return ErrorInfo.MERR_INVALID_PARAM.getValue();
//        }
        if (handle == 0L) {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
        return nativeUpdateFaceFeature(handle, faceFeatureInfo);
    }

    /**
     *  获取人脸特征
     * @param searchId 人脸Id
     * @param faceFeatureInfo 返回人脸信息
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int getFaceFeature(int searchId, FaceFeatureInfo faceFeatureInfo) {
        if (faceFeatureInfo == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        if (handle == 0L) {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }

        return nativeGetFaceFeature(handle, searchId, faceFeatureInfo);
    }

    /**
     * 获取注册人脸数量
     * @param faceSearchCount 返回注册人脸数量
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int getFaceCount(FaceSearchCount faceSearchCount) {
        if (faceSearchCount == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        if (handle == 0L) {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
        return nativeGetFaceCount(handle, faceSearchCount);
    }

    /**
     * 获取年龄信息，需要在调用{@link #process(byte[], int, int, ImageFormat, List, FunctionConfiguration)}后调用
     *
     * @param ageInfoList 年龄信息列表，用于输出年龄信息数据
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int getAge(List<AgeInfo> ageInfoList) {

        if (ageInfoList == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        ageInfoList.clear();
        if (handle != 0L) {
            AgeInfo[] faceInfoArray = new AgeInfo[detectFaceMaxNum];
            int errorCode = nativeGetAge(handle, faceInfoArray);
            for (AgeInfo ageInfo : faceInfoArray) {
                if (ageInfo == null) break;
                ageInfoList.add(ageInfo);
            }

            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }

    }

    /**
     * 获取性别信息，需要在调用{@link #process(byte[], int, int, ImageFormat, List, FunctionConfiguration)}后调用
     *
     * @param genderInfoList 性别信息列表，用于输出性别信息数据
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int getGender(List<GenderInfo> genderInfoList) {
        if (genderInfoList == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        genderInfoList.clear();
        if (handle != 0L) {
            GenderInfo[] genderInfos = new GenderInfo[detectFaceMaxNum];
            int errorCode = nativeGetGender(handle, genderInfos);
            for (GenderInfo genderInfo : genderInfos) {
                if (genderInfo == null) break;
                genderInfoList.add(genderInfo);
            }

            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }

    }

    /**
     * 获取新的RGB活体信息对象，需要在调用{@link #process(byte[], int, int, ImageFormat, List, FunctionConfiguration)}后调用
     *
     * @param livenessInfoList 创建一个新的RGB活体信息对象
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int getLiveness(List<LivenessInfo> livenessInfoList) {
        if (livenessInfoList == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        livenessInfoList.clear();
        if (handle != 0L) {
            LivenessInfo[] livenessInfos = new LivenessInfo[detectFaceMaxNum];
            int errorCode = nativeGetLiveness(handle, livenessInfos);
            for (LivenessInfo livenessInfo : livenessInfos) {
                if (livenessInfo == null) break;
                livenessInfoList.add(livenessInfo);
            }
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * 获取新的IR活体信息对象，需要在调用{@link #processIr(byte[], int, int, ImageFormat, List, FunctionConfiguration)}后调用
     *
     * @param livenessInfoList 创建一个新的IR活体信息对象
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int getLivenessIr(List<IrLivenessInfo> livenessInfoList) {
        if (livenessInfoList == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        livenessInfoList.clear();
        if (handle != 0L) {
            IrLivenessInfo[] livenessInfos = new IrLivenessInfo[detectFaceMaxNum];
            int errorCode = nativeGetLivenessIr(handle, livenessInfos);
            for (IrLivenessInfo livenessInfo : livenessInfos) {
                if (livenessInfo == null) break;
                livenessInfoList.add(livenessInfo);
            }
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * 获取新的口罩信息对象，需要在调用{@link #process(byte[], int, int, ImageFormat, List, FunctionConfiguration)}后调用
     *
     * @param maskInfoList 创建一个新的IR活体信息对象
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int getMask(List<MaskInfo> maskInfoList) {
        if (maskInfoList == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        maskInfoList.clear();
        if (handle != 0L) {
            MaskInfo[] maskInfos = new MaskInfo[detectFaceMaxNum];
            int errorCode = nativeGetMask(handle, maskInfos);
            for (MaskInfo maskInfo : maskInfos) {
                if (maskInfo == null) break;
                maskInfoList.add(maskInfo);
            }
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }

    /**
     * 获取版本信息
     *
     * @return 返回版本信息
     */
    public VersionInfo getVersion() {
        VersionInfo versionInfo = new VersionInfo();
        nativeGetVersion(versionInfo);
        return versionInfo;
    }

    /**
     * 采集设备信息（可离线）
     *
     * @param activeDeviceInfo 创建一个新的采集设备信息对象
     * @return 错误码，详细见({@link ErrorInfo})
     */
    public int getActiveDeviceInfo(ActiveDeviceInfo activeDeviceInfo) {
        if (activeDeviceInfo == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        return nativeGetActiveDeviceInfo(activeDeviceInfo);
    }


    /**
     * 图像质量检测
     *
     * @param imageInfo    图像信息
     * @param faceInfo     人脸列表，传入后赋值
     * @param mask         是否戴口罩，仅支持传入1、0、-1，戴口罩 1，否则认为未戴口罩
     * @param imageQuality 输出图像质量信息
     * @return @return 错误码，详细见({@link ErrorInfo})
     */
    public int imageQualityDetect(ImageInfo imageInfo, FaceInfo faceInfo, int mask, ImageQuality imageQuality) {
        if (imageInfo == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        return imageQualityDetect(imageInfo.getImageData(), imageInfo.getWidth(), imageInfo.getHeight(), imageInfo.getImageFormat(), faceInfo, mask, imageQuality);
    }

    /**
     * 图像质量检测
     *
     * @param data         图像数据
     * @param width        图片宽度，为4的倍数
     * @param height       图片高度，YUYV/I420/NV21/NV12格式为2的倍数；BGR24/GRAY格式无限制
     * @param imageFormat  图像的颜色空间格式
     * @param faceInfo     人脸信息
     * @param mask         是否带口罩，仅支持传入1、0、-1，戴口罩 1，否则认为未戴口罩
     * @param imageQuality 输出图像质量信息
     * @return @return 错误码，详细见({@link ErrorInfo})
     */
    private int imageQualityDetect(byte[] data, int width, int height, ImageFormat imageFormat, FaceInfo faceInfo, int mask, ImageQuality imageQuality) {
        if (data == null || imageFormat == null || faceInfo == null || imageQuality == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(data, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }

        if (handle != 0L) {

            int errorCode = nativeImageQualityDetect(handle, data, width, height, imageFormat.getValue(), faceInfo.getRect(), faceInfo.getOrient(), faceInfo.getFaceData(), FaceInfo.FACE_DATA_SIZE, mask, imageQuality);

            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }
    }


    /**
     * 图像质量检测
     *
     * @param imageInfoEx  图像信息
     * @param faceInfo     人脸列表，传入后赋值
     * @param mask         是否戴口罩，仅支持传入1、0、-1，戴口罩 1，否则认为未戴口罩
     * @param imageQuality 输出图像质量信息
     * @return @return 错误码，详细见({@link ErrorInfo})
     */
    public int imageQualityDetect(ImageInfoEx imageInfoEx, FaceInfo faceInfo, int mask, ImageQuality imageQuality) {
        if (imageInfoEx == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }
        return imageQualityDetect(imageInfoEx.getImageDataPlanes(), imageInfoEx.getImageStrides(), imageInfoEx.getWidth(), imageInfoEx.getHeight(), imageInfoEx.getImageFormat()
                , faceInfo, mask, imageQuality);
    }

    /**
     * @param imageDataPlanes 人脸图像通道
     * @param imageStrides    步长
     * @param width           图片宽度，为4的倍数
     * @param height          图片高度，YUYV/I420/NV21/NV12格式为2的倍数；BGR24/GRAY/DEPTH_U16格式无限制
     * @param imageFormat     图像的颜色空间格式
     * @param mask            是否戴口罩，仅支持传入1、0、-1，戴口罩 1，否则认为未戴口罩
     * @param faceInfo        人脸列表，传入后赋值
     * @param imageQuality    输出图像质量信息
     * @return @return 错误码，详细见({@link ErrorInfo})
     */
    private int imageQualityDetect(byte[][] imageDataPlanes, int[] imageStrides, int width, int height, ImageFormat imageFormat, FaceInfo faceInfo, int mask, ImageQuality imageQuality) {
        if (faceInfo == null
                || imageDataPlanes == null
                || imageStrides == null
                || imageFormat == null
                || imageQuality == null) {
            return ErrorInfo.MERR_INVALID_PARAM.getValue();
        }

        if (width < 1 || height < 1) {
            return ErrorInfo.MERR_ASF_IMAGE_WIDTH_HEIGHT_NOT_SUPPORT.getValue();
        }

        if (!checkImageData(imageDataPlanes, imageStrides, width, height, imageFormat)) {
            return ErrorInfo.MERR_ASF_EX_INVALID_IMAGE_INFO.getValue();
        }

        if (handle != 0L) {
            int errorCode = nativeImageQualityDetectEx(handle, imageDataPlanes, imageStrides, width, height, imageFormat.getValue(), faceInfo.getRect(), faceInfo.orient, faceInfo.faceData, FaceInfo.FACE_DATA_SIZE, mask, imageQuality);
            return errorCode;
        } else {
            return ErrorInfo.MERR_BAD_STATE.getValue();
        }

    }

    private boolean checkImageData(byte[] data, int width, int height, ImageFormat imageFormat) {
        switch (imageFormat) {
            case CP_PAF_NV21:
            case CP_PAF_NV12:
            case CP_PAF_I420:
                if (data.length != width * height * 3 / 2) {
                    return false;
                }
                break;
            case CP_PAF_RGB24:
                if (data.length != width * height * 3) {
                    return false;
                }
                break;
            case CP_PAF_YUYV:
                if (data.length != width * height * 2) {
                    return false;
                }
                break;
            case CP_PAF_GRAY: {
                if (data.length != width * height) {
                    return false;
                }
                break;
            }
            case CP_PAF_DEPTH_U16: {
                if (data.length != width * height * 2) {
                    return false;
                }
                break;
            }
        }
        return true;
    }

    private boolean checkImageData(byte[][] planes, int[] strides, int width, int height, ImageFormat imageFormat) {
        if (planes != null && strides != null) {
            if (planes.length != strides.length) {
                return false;
            } else {
                byte[][] var3 = planes;
                int var4 = planes.length;

                for (int var5 = 0; var5 < var4; ++var5) {
                    byte[] plane = var3[var5];
                    if (plane == null || plane.length == 0) {
                        return false;
                    }
                }

                switch (imageFormat) {
                    case CP_PAF_NV21:
                        return ((height & 1) == 0)
                                && planes.length == 2
                                && planes[0].length == planes[1].length * 2
                                && planes[0].length == (strides[0] * height)
                                && planes[1].length == (strides[1] * height / 2);
                    case CP_PAF_RGB24:
                    case CP_PAF_GRAY:
                    case CP_PAF_DEPTH_U16:
                        return planes.length == 1
                                && planes[0].length == strides[0] * height;
                    default:
                        return false;
                }
            }
        } else {
            return false;
        }
    }


}
